"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
exports.isImageImport = isImageImport;
exports.default = transformSource;
var _swc = require("../../swc");
var _options = require("../../swc/options");
var _utils = require("../../utils");
function isClientComponent(importSource, pageExtensions) {
    return new RegExp(`\\.client(\\.(${pageExtensions.join('|')}))?`).test(importSource);
}
function isServerComponent(importSource, pageExtensions) {
    return new RegExp(`\\.server(\\.(${pageExtensions.join('|')}))?`).test(importSource);
}
function isNextComponent(importSource) {
    return importSource.includes('next/link') || importSource.includes('next/image');
}
function isImageImport(importSource) {
    // TODO: share extension with next/image
    // TODO: add other static assets, jpeg -> jpg
    return [
        'jpg',
        'jpeg',
        'png',
        'webp',
        'avif'
    ].some((imageExt)=>importSource.endsWith('.' + imageExt)
    );
}
async function parseImportsInfo(resourcePath, source, imports, isClientCompilation, pageExtensions) {
    const opts = (0, _options).getBaseSWCOptions({
        filename: resourcePath,
        globalWindow: isClientCompilation
    });
    const ast = await (0, _swc).parse(source, {
        ...opts.jsc.parser,
        isModule: true
    });
    const { body  } = ast;
    const beginPos = ast.span.start;
    let transformedSource = '';
    let lastIndex = 0;
    let defaultExportName;
    for(let i = 0; i < body.length; i++){
        const node = body[i];
        switch(node.type){
            case 'ImportDeclaration':
                {
                    const importSource = node.source.value;
                    if (!isClientCompilation) {
                        if (!(isClientComponent(importSource, pageExtensions) || isNextComponent(importSource) || isImageImport(importSource))) {
                            continue;
                        }
                        const importDeclarations = source.substring(lastIndex, node.source.span.start - beginPos);
                        transformedSource += importDeclarations;
                        transformedSource += JSON.stringify(`${node.source.value}?flight`);
                    } else {
                        // For the client compilation, we skip all modules imports but
                        // always keep client components in the bundle. All client components
                        // have to be imported from either server or client components.
                        if (!(isClientComponent(importSource, pageExtensions) || isServerComponent(importSource, pageExtensions) || // Special cases for Next.js APIs that are considered as client
                        // components:
                        isNextComponent(importSource) || isImageImport(importSource))) {
                            continue;
                        }
                    }
                    lastIndex = node.source.span.end - beginPos;
                    imports.push(`require(${JSON.stringify(importSource)})`);
                    continue;
                }
            case 'ExportDefaultDeclaration':
                {
                    const def = node.decl;
                    if (def.type === 'Identifier') {
                        defaultExportName = def.name;
                    } else if (def.type === 'FunctionExpression') {
                        defaultExportName = def.identifier.value;
                    }
                    break;
                }
            case 'ExportDefaultExpression':
                const exp = node.expression;
                if (exp.type === 'Identifier') {
                    defaultExportName = exp.value;
                }
                break;
            default:
                break;
        }
    }
    if (!isClientCompilation) {
        transformedSource += source.substring(lastIndex);
    }
    return {
        source: transformedSource,
        defaultExportName
    };
}
async function transformSource(source) {
    const { client: isClientCompilation , pageExtensions: pageExtensionsJson  } = this.getOptions();
    const { resourcePath  } = this;
    const pageExtensions = JSON.parse(pageExtensionsJson);
    if (typeof source !== 'string') {
        throw new Error('Expected source to have been transformed to a string.');
    }
    if (resourcePath.includes('/node_modules/')) {
        return source;
    }
    const imports = [];
    const { source: transformedSource , defaultExportName  } = await parseImportsInfo(resourcePath, source, imports, isClientCompilation, (0, _utils).getRawPageExtensions(pageExtensions));
    /**
   * For .server.js files, we handle this loader differently.
   *
   * Server compilation output:
   *   export default function ServerComponent() { ... }
   *   export const __rsc_noop__ = () => { ... }
   *   ServerComponent.__next_rsc__ = 1
   *   ServerComponent.__webpack_require__ = __webpack_require__
   *
   * Client compilation output:
   *   The function body of Server Component will be removed
   */ const noop = `export const __rsc_noop__=()=>{${imports.join(';')}}`;
    let defaultExportNoop = '';
    if (isClientCompilation) {
        defaultExportNoop = `export default function ${defaultExportName || 'ServerComponent'}(){}\n${defaultExportName || 'ServerComponent'}.__next_rsc__=1;`;
    } else {
        if (defaultExportName) {
            // It's required to have the default export for pages. For other components, it's fine to leave it as is.
            defaultExportNoop = `${defaultExportName}.__next_rsc__=1;${defaultExportName}.__webpack_require__=__webpack_require__;`;
        }
    }
    const transformed = transformedSource + '\n' + noop + '\n' + defaultExportNoop;
    return transformed;
}

//# sourceMappingURL=next-flight-server-loader.js.map