import React, { useState } from "react";
import { Col, Form } from "react-bootstrap";
import { FieldValues, useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as yup from "yup";
import { toast } from "react-toastify";
import { ContactDataTypes } from "../common/interfaces";
import { postRequest } from "../common/API";
import variables from "../common/variables";

const validationSchema = yup
  .object({
    name: yup.string().required("Name is required"),
    email: yup
      .string()
      .email("Must be a valid email")
      .required("Email is required"),
    company_name: yup.string().required("Company name is required"),
  })
  .required();

const PortfolioContactForm: React.FC = () => {
  const [loading, setLoading] = useState<boolean>(false);

  const {
    register,
    handleSubmit,
    formState: { errors },
    reset,
  } = useForm({
    resolver: yupResolver(validationSchema),
  });
  const onSubmit = async (data: FieldValues) => {
    setLoading(true);
    try {
      let postData: ContactDataTypes | any = {};
      postData.name = data.name;
      postData.company = data.company_name;
      postData.email = data.email;
      postData.phone = data.phone;
      postData.message = data.message;
      // send data to backend
      const res = await postRequest(variables.apiUrls.postContact, postData);
      if (res?.data.statusCode === variables.statusCodes.successCode) {
        toast.success(res.data.message);
        reset();
      } else if (
        res?.data.statusCode === variables.statusCodes.validationError
      ) {
        const msg = Object.values(res.data.message).flat(Infinity).join("");
        toast.error(msg);
      } else {
        toast.error("Something went wrong. Please try again!");
      }
      //reset form field

      setLoading(false);
    } catch (err) {
      toast.error("Something went wrong. Please try again!");
      setLoading(false);
    }
  };

  return (
    <Form onSubmit={handleSubmit(onSubmit)}>
      <Form.Group className="form_control flex-wrap d-sm-flex py-3">
        <Col lg={6} md={12} className="pe-lg-2 pe-0">
          <Form.Label>Your Name*</Form.Label>
          <Form.Control
            className="input_item is-invalid"
            type="text"
            {...register("name")}
            style={{
              border: errors.name?.message ? "1px solid #dc3545" : "none",
            }}
          />
          <div className="invalid-feedback f-12">{errors.name?.message}</div>
        </Col>

        <Col lg={6} md={12} className="ps-lg-2 ps-0">
          <Form.Label>Your Email*</Form.Label>
          <Form.Control
            className="input_item is-invalid"
            type="email"
            style={{
              border: errors.email?.message ? "1px solid #dc3545" : "none",
            }}
            {...register("email")}
          />
          <div className="invalid-feedback f-12">{errors.email?.message}</div>
        </Col>

        <Col lg={6} md={12} className="pe-lg-2 pe-0">
          <Form.Label>Your Company*</Form.Label>
          <Form.Control
            className="input_item is-invalid"
            type="text"
            {...register("company_name")}
            style={{
              border: errors.company_name?.message
                ? "1px solid #dc3545"
                : "none",
            }}
          />
          <div className="invalid-feedback f-12">
            {errors.company_name?.message}
          </div>
        </Col>

        <Col lg={6} md={12} className="ps-lg-2 ps-0">
          <Form.Label>Your Phone</Form.Label>
          <Form.Control
            className="input_item"
            type="text"
            {...register("phone")}
          />
        </Col>
      </Form.Group>
      {/* <Form.Group className="pb-4">
        <Form.Check
          className="check_box_section"
          inline
          label="Save my name, email, and website in this browser for the
    next time I comment."
          type="checkbox"
        />
      </Form.Group> */}

      <Form.Group className="form_control mb-4">
        <Form.Label>Leave a Reply</Form.Label>
        <Form.Control
          className="text_area_section"
          as="textarea"
          rows={6}
          {...register("message")}
        />
      </Form.Group>

      <button className="btnStyle d-block d-lg-inline-block" type="submit">
        {loading && (
          <span
            className="spinner-border spinner-border-sm"
            role="status"
            aria-hidden="true"
            style={{ marginRight: "5px" }}
          ></span>
        )}
        Send Message
      </button>
    </Form>
  );
};

export default PortfolioContactForm;
