import { useRouter } from "next/router";
import { createContext, useState, useEffect, useContext } from "react";
import Immerser from "immerser";
export const UiContext = createContext();
export const defaultUiState = {
  fullScreenMenuOpen: false,
  overlayLeftRef: null,
  setting: {
    site_name: "SoftwindTech Ltd",
    site_title: "SoftwindTech is a leading digital marketing agency",
    site_description:
      "SoftwindTech is a leading digital marketing agency, offers holistic results driven marketing solution to businesses worldwide.",
    site_logo: "",
    menus: {
      main: [],
      footer: [],
      mobile: [],
    },
  },
};

function overlayImmer() {
  try {
    const immerserInstance = new Immerser({
      // this option will be overridden by options
      // passed in data-immerser-layer-config attribute in each layer
      solidClassnameArray: [
        {
          left_scroll: "left_scroll_color_white",
        },
      ],
      hasToUpdateHash: false,
      // fromViewportWidth: 1024,
      // pagerLinkActiveClassname: "pager__link--active",
      scrollAdjustThreshold: 0,
      scrollAdjustDelay: 600,
      onInit(immerser) {
        // callback on init
      },
      onBind(immerser) {
        // callback on bind
        document.querySelector(".scroll_btn")?.classList.remove("toTop");
      },
      onUnbind(immerser) {
        // callback on unbind
        document.querySelector(".scroll_btn")?.classList.add("toTop");
      },
      onDestroy(immerser) {
        // callback on destroy
      },
      onActiveLayerChange(activeIndex, immerser) {
        // callback on active layer change
      },
    });
    window.immerserInstance = immerserInstance;
  } catch (e) {}
}

export const UiContextProvider = ({ children }) => {
  const [UiState, setUiState] = useState(defaultUiState);

  const router = useRouter();
  const onRouteChangeDone = (url, { shallow }) => {
    overlayImmer();
    setUiState((prev) => {
      return { ...prev, fullScreenMenuOpen: false };
    });
  };
  const onRouteChangeStart = (url, { shallow }) => {
    window.immerserInstance?.destroy();
  };

  const onRouteChangeError = (err, url, { shallow }) => {
    if (err.cancelled) {
      overlayImmer();
    }
  };

  useEffect(() => {
    router.events.on("routeChangeComplete", onRouteChangeDone);
    router.events.on("routeChangeStart", onRouteChangeStart);
    router.events.on("routeChangeError", onRouteChangeError);
    return () => {
      router.events.off("routeChangeComplete", onRouteChangeDone);
      router.events.off("routeChangeStart", onRouteChangeStart);
      router.events.off("routeChangeError", onRouteChangeError);
    };
  }, [router]);

  useEffect(() => {
    if (localStorage.getItem(`UiData`)) {
      setUiState(JSON.parse(localStorage.getItem(`UiData`)));
    }
    return () => {};
  }, []);

  useEffect(() => {
    if (!window.immerserInstance) {
      overlayImmer();
    }
  }, []);

  return (
    <UiContext.Provider value={[UiState, setUiState]}>
      {children}
    </UiContext.Provider>
  );
};

export const useUiState = () => {
  const [UiState, setUiState] = useContext(UiContext);
  return [UiState, setUiState];
};

export const useUiStateModifier = () => {
  const [UiState, setUiState] = useContext(UiContext);
  const setFullScreenMenuOpen = (show) => {
    setUiState((prev) => {
      return { ...prev, fullScreenMenuOpen: show };
    });
  };

  return { setFullScreenMenuOpen };
};
